
// ======= Config & Storage Keys =======
const DEFAULT_BASE = "http://45.252.248.148:8010";
const STORE_KEYS = {
  BASE: "base_origin",
  USER: "auth_user",
  XTOKEN: "x_token",
  DEVICES: "devices",        // [{name, device_token}]
  AUTOCALL: "auto_call"      // bool
};

// ======= Utils =======

function buildAuthHeaders(tok){
  const h = {"Content-Type":"application/json"};
  if (tok){ h["X-Token"] = tok; h["Authorization"] = "Bearer " + tok; }
  return h;
}

function sleep(ms){ return new Promise(r => setTimeout(r, ms)); }
function cleanNumber(raw){ return String(raw||"").replace(/[^\d+]/g, ""); }
function toVN(raw){
  const s = cleanNumber(raw);
  if (s.startsWith("+84") && s.length>=12) return "0" + s.slice(-9);
  if (s.startsWith("0084") && s.length>=13) return "0" + s.slice(-9);
  if (/^0\d{9}$/.test(s)) return s;
  return s;
}
async function getCfg(){
  const cfg = await chrome.storage.sync.get({
    [STORE_KEYS.BASE]: DEFAULT_BASE,
    [STORE_KEYS.USER]: "",
    [STORE_KEYS.XTOKEN]: "",
    [STORE_KEYS.DEVICES]: [],
    [STORE_KEYS.AUTOCALL]: false,
  });
  return cfg;
}
function buildOrigin(baseLike){
  try { return new URL(baseLike).origin; }
  catch (e) {
    return String(baseLike||"").replace(/\/+$/,"");
  }
}
async function apiPost(pathOrUrl, body){
  const cfg = await getCfg();
  const origin = buildOrigin(cfg[STORE_KEYS.BASE] || DEFAULT_BASE);
  const url = pathOrUrl.startsWith("http") ? pathOrUrl : origin + pathOrUrl;
  const headers = {"Content-Type":"application/json"};
  const tok = (cfg[STORE_KEYS.XTOKEN]||"").trim();
  if (tok) headers["X-Token"] = tok;
  const res = await fetch(url, {method:"POST", headers, body: JSON.stringify(body||{})});
  if (!res.ok){
    let t = await res.text().catch(()=> "");
    throw new Error(`HTTP ${res.status}: ${t||res.statusText}`);
  }
  try { return await res.json(); } catch(_){ return await res.text(); }
}

// Try multiple endpoints for each action (robust to server paths)



async function callNumber({device_token, number, sim}){
  const cfg = await getCfg();
  const base = buildOrigin(cfg[STORE_KEYS.BASE] || DEFAULT_BASE);
  const user = cfg[STORE_KEYS.USER] || "";
  const tok  = (cfg[STORE_KEYS.XTOKEN]||"").trim();
  const headers = {"Content-Type":"application/json"};
  if (tok) headers["X-Token"] = tok;

  const cleanNum = toVN(number);
  const noSimBody   = { device_token, number: cleanNum, autoCall: !!cfg[STORE_KEYS.AUTOCALL] };
  const withSimBody = { device_token, number: cleanNum, sim: Number(sim)||1, autoCall: !!cfg[STORE_KEYS.AUTOCALL] };

  const origin = base.replace(/\/+$/,"");
  const urls = [
    `${origin}/api/call/dial?user=${encodeURIComponent(user)}`,
    `${origin}/call/dial?user=${encodeURIComponent(user)}`,
    `${origin}/api/calls/dial?user=${encodeURIComponent(user)}`,
    `${origin}/calls/dial?user=${encodeURIComponent(user)}`,
    // Legacy fallbacks
    `${origin}/devices/call`,
    `${origin}/api/devices/call`,
    `${origin}/call/dial`,
    `${origin}/api/call/dial`,
    `${origin}/calls/dial`,
    `${origin}/api/calls/dial`
  ];

  let lastErr;
  for (const u of urls){
    try{
      // prefer body without sim
      let r = await fetch(u, {method:"POST", headers, body: JSON.stringify(noSimBody)});
      if (r.ok) return await r.text();
      // fallback with sim
      r = await fetch(u, {method:"POST", headers, body: JSON.stringify(withSimBody)});
      if (r.ok) return await r.text();
      lastErr = new Error(`HTTP ${r.status}: ${await r.text()}`);
    }catch(e){ lastErr = e; }
  }
  throw lastErr || new Error("No working call endpoint");
}


// Test whether a device token is reachable/active (non-intrusive)
// Strategy: prefer harmless actions (ping/echo/clipboard) over call.
// Success if any endpoint returns 2xx.
async function testDevice({device_token}){
  const cfg = await getCfg();
  const origin = buildOrigin(cfg[STORE_KEYS.BASE] || DEFAULT_BASE);
  const tok = (cfg[STORE_KEYS.XTOKEN]||"").trim();
  const headers = {"Content-Type":"application/json"};
  if (tok) headers["X-Token"] = tok;

  const payloads = [
    {url:`${origin}/devices/ping`, body:{device_token}},
    {url:`${origin}/api/devices/ping`, body:{device_token}},
    {url:`${origin}/devices/echo`, body:{device_token, text:"PING"}},
    {url:`${origin}/api/devices/echo`, body:{device_token, text:"PING"}},
    {url:`${origin}/devices/clipboard`, body:{device_token, text:"PING"}}, // harmless
    {url:`${origin}/api/devices/clipboard`, body:{device_token, text:"PING"}},
    {url:`${origin}/device/clipboard`, body:{device_token, text:"PING"}},
    {url:`${origin}/api/device/clipboard`, body:{device_token, text:"PING"}}
  ];

  let lastErr;
  for (const {url, body} of payloads){
    try{
      const r = await fetch(url, {method:"POST", headers, body: JSON.stringify(body)});
      if (!r.ok) { lastErr = new Error(`HTTP ${r.status}: ${await r.text()}`); continue; }
      // accept json or text
      let data = null;
      try { data = await r.json(); } catch(_){ data = await r.text(); }
      return {ok:true, url, data};
    }catch(e){ lastErr = e; }
  }
  throw lastErr || new Error("No working test endpoint");
}


async function testDevice({device_token}){
  const cfg = await getCfg();
  const base = buildOrigin(cfg[STORE_KEYS.BASE] || DEFAULT_BASE);
  const user = cfg[STORE_KEYS.USER] || "";
  const tok  = (cfg[STORE_KEYS.XTOKEN]||"").trim();
  const headers = {"Content-Type":"application/json"};
  if (tok) headers["X-Token"] = tok;

  const b = base.replace(/\/+$/,"");
  const variants = [
    `${b}/api/call/dial?user=${encodeURIComponent(user)}`,
    `${b}/call/dial?user=${encodeURIComponent(user)}`,
    `${b}/api/calls/dial?user=${encodeURIComponent(user)}`,
    `${b}/calls/dial?user=${encodeURIComponent(user)}`
  ];

  // minimal body to avoid real call; expect 4xx (400/422) as "reachable"
  const body = JSON.stringify({ device_token });

  let lastErr;
  for (const u of variants){
    try{
      const r = await fetch(u, {method:"POST", headers, body});
      if (r.status === 404){ lastErr = new Error("404 Not Found"); continue; }
      if (r.status === 401 || r.status === 403){ lastErr = new Error("401/403 Unauthorized"); continue; }
      // 2xx or 4xx other than 401/403/404 => endpoint reachable
      return {ok:true, url:u, code:r.status, text: await r.text()};
    }catch(e){ lastErr = e; }
  }
  throw lastErr || new Error("No working test endpoint");
}

// ======= Message Router from content/popup =======
chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
  (async () => {
    try {
      if (msg && msg.type === "ccp_call"){
        const r = await callNumber(msg.payload);
        sendResponse({ok:true, data:r}); return;
      }
      if (msg && msg.type === "ccp_hangup"){
        const r = await hangup(msg.payload);
        sendResponse({ok:true, data:r}); return;
      }
      if (msg && msg.type === "ccp_paste"){
        const r = await pasteNumber(msg.payload);
        sendResponse({ok:true, data:r}); return;
      }
      if (msg && msg.type === "ccp_test_device"){ const r = await testDevice(msg.payload); sendResponse({ok:true, data:r}); return; }
      if (msg && msg.type === "ccp_test_device"){ const r = await testDevice(msg.payload); sendResponse({ok:true, data:r}); return; }
      if (msg && msg.type === "ccp_get_cfg"){
        sendResponse({ok:true, data: await getCfg()}); return;
      }
      if (msg && msg.type === "ccp_toggle_panel"){
        // toggle flag in storage so content.js can draw/hide
        const st = await chrome.storage.local.get({panel_enabled:false});
        const next = !st.panel_enabled;
        await chrome.storage.local.set({panel_enabled: next});
        sendResponse({ok:true, enabled: next}); return;
      }
    } catch(e){
      sendResponse({ok:false, error: e.message || String(e)});
    }
  })();
  return true; // keep channel open for async
});

// Toolbar click -> toggle panel

chrome.action.onClicked.addListener(async (tab) => {
  if (!tab || !tab.id) return;
  try { await chrome.tabs.sendMessage(tab.id, {type:"ccp_toggle_panel"}); return; } catch(_){ }
  try { await chrome.scripting.executeScript({target:{tabId:tab.id}, files:["content.js"]}); await chrome.tabs.sendMessage(tab.id, {type:"ccp_show_panel"}); return; } catch(_){ }
  try{
    if (!tab || !tab.id) return;
    try{
      await chrome.tabs.sendMessage(tab.id, {type:"ccp_toggle_panel"});
      return;
    }catch(_){}
    // Fallback: inject content.js then toggle
    await chrome.scripting.executeScript({target:{tabId:tab.id}, files:["content.js"]});
    await chrome.tabs.sendMessage(tab.id, {type:"ccp_toggle_panel"});
  }catch(e){
    console.warn("Toggle failed:", e);
  }
});
